/**
* \file: e_sdc_ecm.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: sdc-ecm-engine
*
* \author: Kirill Marinushkin (kmarinushkin@de.adit-jv.com)
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <openssl/engine.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include "e_sdc_ecm.h"
#include "e_sdc_ecm_dbus.h"
#include "e_sdc_ecm_err.h"
#include "e_sdc_ecm_log.h"

static const char sdc_ecm_id[] = "sdc-ecm";
static const char sdc_ecm_name[] = "Asymmetric interaction with SDC and ECM";

/* RSA methods */
static int sdc_ecm_rsa_pub_enc(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding);
static int sdc_ecm_rsa_pub_dec(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding);
static int sdc_ecm_rsa_priv_enc(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding);
static int sdc_ecm_rsa_priv_dec(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding);
static int sdc_ecm_rsa_sign(int type,
    const unsigned char *m, unsigned int m_length,
    unsigned char *sigret, unsigned int *siglen, const RSA *rsa);
static int sdc_ecm_rsa_verify(int type,
    const unsigned char *m, unsigned int m_length,
    const unsigned char *sigbuf, unsigned int siglen, const RSA *rsa);
static int sdc_ecm_rsa_init(RSA *rsa);
static int sdc_ecm_rsa_finish(RSA *rsa);

static RSA_METHOD sdc_ecm_rsa = {
    .name = "sdc-ecm RSA",
    .rsa_pub_enc = sdc_ecm_rsa_pub_enc,
    .rsa_pub_dec = sdc_ecm_rsa_pub_dec,
    .rsa_priv_enc = sdc_ecm_rsa_priv_enc,
    .rsa_priv_dec = sdc_ecm_rsa_priv_dec,
    .rsa_mod_exp = NULL,
    .bn_mod_exp = NULL,
    .init = sdc_ecm_rsa_init,
    .finish = sdc_ecm_rsa_finish,
    .flags = RSA_FLAG_THREAD_SAFE |
             RSA_FLAG_SIGN_VER,
    .rsa_sign = sdc_ecm_rsa_sign,
    .rsa_verify = sdc_ecm_rsa_verify,
};

static int sdc_ecm_rsa_dbus_ctx_ind = -1;

/* PKEY methods */
static const int sdc_ecm_pkey_nids[] = { EVP_PKEY_RSA, 0 };

struct sdc_ecm_pkey_ctx_t {
    struct sdc_ecm_dbus_ctx *dbus_ctx;
    int md_type;
};

static int sdc_ecm_pkey_init(EVP_PKEY_CTX *ctx);
static int sdc_ecm_pkey_copy(EVP_PKEY_CTX *dst, EVP_PKEY_CTX *src);
static void sdc_ecm_pkey_cleanup(EVP_PKEY_CTX *ctx);
static int sdc_ecm_pkey_ctrl(EVP_PKEY_CTX *ctx, int type, int p1, void *p2);
static int sdc_ecm_pkey_ctrl_str(EVP_PKEY_CTX *ctx, const char *type,
    const char *value);
static int sdc_ecm_pkey_sign_init(EVP_PKEY_CTX *ctx);
static int sdc_ecm_pkey_sign(EVP_PKEY_CTX *ctx,
    unsigned char *sig, size_t *siglen,
    const unsigned char *tbs, size_t tbslen);
static int sdc_ecm_pkey_verify_init(EVP_PKEY_CTX *ctx);
static int sdc_ecm_pkey_verify(EVP_PKEY_CTX *ctx,
    const unsigned char *sig, size_t siglen,
    const unsigned char *tbs, size_t tbslen);
static int sdc_ecm_pkey_encrypt_init(EVP_PKEY_CTX *ctx);
static int sdc_ecm_pkey_encrypt(EVP_PKEY_CTX *ctx,
    unsigned char *out, size_t *outlen,
    const unsigned char *in, size_t inlen);
static int sdc_ecm_pkey_decrypt_init(EVP_PKEY_CTX *ctx);
static int sdc_ecm_pkey_decrypt(EVP_PKEY_CTX *ctx,
    unsigned char *out, size_t *outlen,
    const unsigned char *in, size_t inlen);

static EVP_PKEY_METHOD *sdc_ecm_pkey_rsa = NULL;

static int sdc_ecm_pkey_meth_init(void)
{
    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_pkey_rsa = EVP_PKEY_meth_new(sdc_ecm_pkey_nids[0], 0);
    if (!sdc_ecm_pkey_rsa) {
        SDC_ECMerr(SDC_ECM_F_PKEY_METH_INIT, SDC_ECM_R_NO_MEM);
        return 0;
    }

    EVP_PKEY_meth_set_init(sdc_ecm_pkey_rsa, sdc_ecm_pkey_init);
    EVP_PKEY_meth_set_copy(sdc_ecm_pkey_rsa, sdc_ecm_pkey_copy);
    EVP_PKEY_meth_set_cleanup(sdc_ecm_pkey_rsa, sdc_ecm_pkey_cleanup);
    EVP_PKEY_meth_set_ctrl(sdc_ecm_pkey_rsa, sdc_ecm_pkey_ctrl,
        sdc_ecm_pkey_ctrl_str);
    EVP_PKEY_meth_set_sign(sdc_ecm_pkey_rsa,
        sdc_ecm_pkey_sign_init, sdc_ecm_pkey_sign);
    EVP_PKEY_meth_set_verify(sdc_ecm_pkey_rsa,
        sdc_ecm_pkey_verify_init, sdc_ecm_pkey_verify);
    EVP_PKEY_meth_set_encrypt(sdc_ecm_pkey_rsa,
        sdc_ecm_pkey_encrypt_init, sdc_ecm_pkey_encrypt);
    EVP_PKEY_meth_set_decrypt(sdc_ecm_pkey_rsa,
        sdc_ecm_pkey_decrypt_init, sdc_ecm_pkey_decrypt);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static inline void sdc_ecm_pkey_meth_destroy(void)
{
    sdc_ecm_dbg(__FUNCTION__);

    EVP_PKEY_meth_free(sdc_ecm_pkey_rsa);
    sdc_ecm_pkey_rsa = NULL;
}

/* PKEY ASN1 methods */
#define SDC_ECM_ASN1_PUB    EVP_PKEY_RSA

static const int sdc_ecm_pkey_asn1_nids[] = { SDC_ECM_ASN1_PUB, 0 };

static int sdc_ecm_pkey_asn1_pub_cmp(const EVP_PKEY *a, const EVP_PKEY *b);
static int sdc_ecm_pkey_asn1_pub_dec(EVP_PKEY *pk, X509_PUBKEY *pub);
static int sdc_ecm_pkey_asn1_pub_enc(X509_PUBKEY *pub, const EVP_PKEY *pk);

static EVP_PKEY_ASN1_METHOD *sdc_ecm_pkey_asn1_pub = NULL;

static int sdc_ecm_pkey_asn1_meth_init(void)
{
    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_pkey_asn1_pub = EVP_PKEY_asn1_new(sdc_ecm_pkey_asn1_nids[0], 0,
                                              "sdc-ecm-asn1-pub",
                                              "sdc-ecm ASN1 public key");
    if (!sdc_ecm_pkey_asn1_pub) {
        SDC_ECMerr(SDC_ECM_F_PKEY_ASN1_METH_INIT, SDC_ECM_R_NO_MEM);
        return 0;
    }

    EVP_PKEY_asn1_set_public(sdc_ecm_pkey_asn1_pub,
        sdc_ecm_pkey_asn1_pub_dec,
        sdc_ecm_pkey_asn1_pub_enc,
        sdc_ecm_pkey_asn1_pub_cmp,
        NULL,
        NULL,
        NULL);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static inline void sdc_ecm_pkey_asn1_meth_destroy(void)
{
    sdc_ecm_dbg(__FUNCTION__);

    EVP_PKEY_asn1_free(sdc_ecm_pkey_asn1_pub);
    sdc_ecm_pkey_asn1_pub = NULL;
}

/* Non-standard commands */
#define LOAD_CERT_CTRL  ENGINE_CMD_BASE
#define ECM_CHECK_CERT  ENGINE_CMD_BASE + 1

#define DBUS_BUS_TYPE           ENGINE_CMD_BASE + 100
#define DBUS_BUS_NAME           ENGINE_CMD_BASE + 101
#define DBUS_INTERFACE_NAME     ENGINE_CMD_BASE + 102
#define ECM_CHECK_CERT_MODE     ENGINE_CMD_BASE + 103
#define LOGLEVEL                ENGINE_CMD_BASE + 104
#define LOGFILENAME             ENGINE_CMD_BASE + 105

/* Engine context */
enum sdc_ecm_check_cert_mode_t {
    CHECK_CERT_MODE_SERVER = 0,
    CHECK_CERT_MODE_CHAIN,
    CHECK_CERT_MODE_OFF,
};

#define CHECK_CERT_MODE_SERVER_STRING "server"
#define CHECK_CERT_MODE_CHAIN_STRING  "chain"
#define CHECK_CERT_MODE_OFF_STRING    "off_because_run_as_server"

static ENGINE_CMD_DEFN sdc_ecm_cmd_defns[] = {
    { LOAD_CERT_CTRL, "LOAD_CERT_CTRL",
        "Load certificate", 0 },
    { ECM_CHECK_CERT, "ECM_CHECK_CERT",
        "Check certificate using ECM", 0 },
    { DBUS_BUS_TYPE, "DBUS_BUS_TYPE",
        "DBus bus type: system|session",
        ENGINE_CMD_FLAG_STRING },
    { DBUS_BUS_NAME, "DBUS_BUS_NAME",
        "DBus bus name",
        ENGINE_CMD_FLAG_STRING },
    { DBUS_INTERFACE_NAME, "DBUS_INTERFACE_NAME",
        "DBus interface name",
        ENGINE_CMD_FLAG_STRING },
    { ECM_CHECK_CERT_MODE, "ECM_CHECK_CERT_MODE",
        "ECM check certificate mode: " CHECK_CERT_MODE_SERVER_STRING "|"
        CHECK_CERT_MODE_CHAIN_STRING "| " CHECK_CERT_MODE_OFF_STRING,
        ENGINE_CMD_FLAG_STRING },
    { LOGLEVEL, "LOGLEVEL",
        "Log level: 1=errors, 2=info, 3=debug, 4=debug_deep, 0=none",
        ENGINE_CMD_FLAG_NUMERIC },
    { LOGFILENAME, "LOGFILENAME",
        "Log file name, default is " DEF_LOGFILENAME,
        ENGINE_CMD_FLAG_STRING },
    { }
};

struct sdc_ecm_engine_ctx_t {
    enum sdc_ecm_check_cert_mode_t check_cert_mode;
    struct sdc_ecm_dbus_ctx dbus_ctx;
    struct sdc_ecm_key_ctx key_ctx;
};

static struct sdc_ecm_engine_ctx_t sdc_ecm_engine_ctx = {
        .check_cert_mode = CHECK_CERT_MODE_SERVER,
        .dbus_ctx = { },
        .key_ctx = { }
};

int loglevel = SDC_ECM_LOGL_NONE;
char logfilename[PATH_MAX] = "";

#ifdef NO_SDC_PREHASHED_SIGN
/* Digests definition */
#define IMPLEMENT_sdc_ecm_md_funcs(nid, name, digest_len, cblock, ctx_name) \
static int sdc_ecm_##name##_init(EVP_MD_CTX *ctx) \
{ \
    sdc_ecm_ext_dbg(__FUNCTION__); \
 \
    if (!sdc_ecm_dbus_hash_init(ctx, &sdc_ecm_engine_ctx.dbus_ctx, \
        digest_len)) \
        return 0; \
 \
    return name##_Init(ctx->md_data); \
} \
 \
static int sdc_ecm_##name##_update(EVP_MD_CTX *ctx, const void *data, \
        size_t count) \
{ \
    sdc_ecm_ext_dbg(__FUNCTION__); \
 \
    if (!sdc_ecm_dbus_hash_update(ctx, &sdc_ecm_engine_ctx.dbus_ctx, \
        data, count)) \
        return 0; \
 \
    return name##_Update(ctx->md_data, data, count); \
} \
 \
static int sdc_ecm_##name##_final(EVP_MD_CTX *ctx, unsigned char *md) \
{ \
    int rc; \
 \
    sdc_ecm_ext_dbg(__FUNCTION__); \
 \
    rc = name##_Final(md, ctx->md_data); \
 \
    if (!sdc_ecm_dbus_hash_final(ctx, &sdc_ecm_engine_ctx.dbus_ctx, md)) \
        return 0; \
 \
    return rc; \
} \
 \
static int sdc_ecm_##name##_copy(EVP_MD_CTX *to, const EVP_MD_CTX *from) \
{ \
    sdc_ecm_ext_dbg(__FUNCTION__); \
 \
    if (!to->md_data || !from->md_data) \
        return 1; \
 \
    memcpy(to->md_data, from->md_data, sizeof(ctx_name)); \
 \
    if (!sdc_ecm_dbus_hash_copy(to, (EVP_MD_CTX *)from, \
        &sdc_ecm_engine_ctx.dbus_ctx)) \
        return 0; \
 \
    return 1; \
} \
 \
static int sdc_ecm_##name##_cleanup(EVP_MD_CTX *ctx) \
{ \
    sdc_ecm_ext_dbg(__FUNCTION__); \
 \
    if (ctx->md_data) \
        memset(ctx->md_data, 0, sizeof(ctx_name)); \
 \
    return 1; \
} \
 \
static const EVP_MD sdc_ecm_##name##_md = { \
    nid, \
    nid##WithRSAEncryption, \
    digest_len, \
    EVP_MD_FLAG_PKEY_METHOD_SIGNATURE, \
    sdc_ecm_##name##_init, \
    sdc_ecm_##name##_update, \
    sdc_ecm_##name##_final, \
    sdc_ecm_##name##_copy, \
    sdc_ecm_##name##_cleanup, \
    EVP_PKEY_NULL_method, \
    cblock, \
    sizeof(EVP_MD *) + sizeof(ctx_name), \
    NULL \
};

/* Digests implementation */
IMPLEMENT_sdc_ecm_md_funcs(NID_sha1, SHA1,
        SHA_DIGEST_LENGTH, SHA_CBLOCK, SHA_CTX)
IMPLEMENT_sdc_ecm_md_funcs(NID_sha256, SHA256,
        SHA256_DIGEST_LENGTH, SHA256_CBLOCK, SHA256_CTX)
IMPLEMENT_sdc_ecm_md_funcs(NID_sha512, SHA512,
        SHA512_DIGEST_LENGTH, SHA512_CBLOCK, SHA512_CTX)

/* Explicitly don't support NID_md5 and NID_sha384 due to usage risks */
static int sdc_ecm_md_nids[] = { NID_sha1, NID_sha256, NID_sha512, 0 };
#endif /* NO_SDC_PREHASHED_SIGN */

static int update_logfilename(const char *n)
{
    if (!n)
        return 0;

    strncpy(logfilename, n, sizeof(logfilename));

    /* return 1 if succeed or 0 if error */
    return 1;
}

static inline int update_loglevel(int l)
{
    loglevel = l;

    sdc_ecm_inf("Loglevel : %d", l);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static void report_err_stunnel_conf(int func_code)
{
    sdc_ecm_dbg(__FUNCTION__);

    /* Forbid operation:
     * if stunnel does this call - means it will skip the
     * certificate check with ECM.
     * Stunnel configuration should be modified to avoid this call */
    sdc_ecm_err("Operation denied: invalid stunnel configuration");
    sdc_ecm_err("Stunnel option key=<key_id> should not be specified");
    sdc_ecm_err("Stunnel option cert=<cert_id> should not be specified");
    SDC_ECMerr(func_code, SDC_ECM_R_INVALID_STUNNEL_CONFIGURATION);
}

static EVP_PKEY *sdc_ecm_load_privkey(ENGINE *e, const char *keyid,
    UI_METHOD *ui_method, void *callback_data)
{
    (void)keyid;
    (void)ui_method;
    (void)callback_data;

    EVP_PKEY *pkey = NULL;

    sdc_ecm_dbg(__FUNCTION__);

    if (!sdc_ecm_engine_ctx.key_ctx.pkey)
        sdc_ecm_engine_ctx.key_ctx.ready = 0;

    /* load private key - only if it is not yet ready */
    if (!sdc_ecm_engine_ctx.key_ctx.ready) {
        int rc = sdc_ecm_dbus_load_key_pair(e, &sdc_ecm_engine_ctx.dbus_ctx,
                    &sdc_ecm_engine_ctx.key_ctx);
        if (!rc) {
            sdc_ecm_err("Failed load client pkey:sdc_iface_load_key_pair");
            SDC_ECMerr(SDC_ECM_F_LOAD_PRIVKEY,
                    SDC_ECM_R_LOAD_KEY_FAILED);

            return NULL;
        }
    }

    pkey = sdc_ecm_engine_ctx.key_ctx.pkey;

    /* user will free the pkey pointer, so forget it to avoid giving the same
     * pointer to several users*/
    sdc_ecm_engine_ctx.key_ctx.pkey = NULL;

    return pkey;
}

static EVP_PKEY *sdc_ecm_load_pubkey(ENGINE *e, const char *keyid,
    UI_METHOD *ui_method, void *callback_data)
{
    (void)e;
    (void)keyid;
    (void)ui_method;
    (void)callback_data;

    sdc_ecm_dbg(__FUNCTION__);

    report_err_stunnel_conf(SDC_ECM_F_LOAD_PUBKEY);

    return NULL;
}

static int sdc_ecm_load_client_cert(ENGINE *e, SSL *ssl,
    STACK_OF(X509_NAME) *ca_dn, X509 **pcert, EVP_PKEY **pkey,
    STACK_OF(X509) **pother, UI_METHOD *ui_method, void *callback_data)
{
    (void)ca_dn;
    (void)pother;
    (void)ui_method;
    (void)callback_data;

    int rc = 0;
    int i;
    X509 *server_cert = NULL;
    STACK_OF(X509) *server_chain = NULL;
    int server_cert_num = 0;
    char logbuf[MAX_INPUT];

    sdc_ecm_dbg(__FUNCTION__);

    /* prepare certificates to verify */
    switch (sdc_ecm_engine_ctx.check_cert_mode) {

    case CHECK_CERT_MODE_SERVER:
        /* send to ECM only server certificate */
        server_cert = SSL_get_peer_certificate(ssl);
        if (!server_cert) {
            sdc_ecm_err("Invalid SSL context: SSL_get_peer_certificate");
            SDC_ECMerr(SDC_ECM_F_LOAD_CLIENT_CERT,
                    SDC_ECM_R_INVALID_SSL_CONTEXT);

            goto out;
        }

        server_cert_num = 1;

        break;

    case CHECK_CERT_MODE_CHAIN:
        /* send to ECM full certificate chain */
        server_chain = SSL_get_peer_cert_chain(ssl);
        if (!server_chain || sk_X509_num(server_chain) < 1) {
            sdc_ecm_err("Invalid SSL context: SSL_get_peer_cert_chain");
            SDC_ECMerr(SDC_ECM_F_LOAD_CLIENT_CERT,
                    SDC_ECM_R_INVALID_SSL_CONTEXT);

            goto out;
        }

        server_cert_num = sk_X509_num(server_chain);

        break;

    case CHECK_CERT_MODE_OFF:
        /* This check_cert_mode exist for the situation, when stunnel runs as
         * server. We are currently inside the function which is only executed,
         * when stunnel runs as client. This is an error. */
        sdc_ecm_err(
            "It is forbidden to turn cert check off when run as client");
        SDC_ECMerr(SDC_ECM_F_LOAD_CLIENT_CERT,
            SDC_ECM_R_FORBIDDEN_TO_TURN_CERT_CHECK_OFF_WHEN_RUN_AS_CLIENT);

        goto out;
    }

    /* send to ECM certificates to check */
    for (i = 0; i < server_cert_num; i++) {
        X509 *cert = server_chain ?
                sk_X509_value(server_chain, i) :
                server_cert;

        sdc_ecm_inf("Sending to Gateway: %s", X509_NAME_oneline(
                X509_get_subject_name(cert),
                logbuf, sizeof(logbuf)));

        rc = sdc_ecm_dbus_check_server_cert(&sdc_ecm_engine_ctx.dbus_ctx,
                                            cert);
        if (!rc) {
            sdc_ecm_err("Server cert check failed: sdc_ecm_load_client_cert");
            SDC_ECMerr(SDC_ECM_F_LOAD_CLIENT_CERT,
                    SDC_ECM_R_GATEWAY_CERT_CHECK_FAILED);

            goto out;
        }
    }

    /* load client certificate - always, even if it is ready */
    rc = sdc_ecm_dbus_load_key_pair(e, &sdc_ecm_engine_ctx.dbus_ctx,
                &sdc_ecm_engine_ctx.key_ctx);
    if (!rc) {
        sdc_ecm_err("Failed load client cert:sdc_iface_load_key_pair");
        SDC_ECMerr(SDC_ECM_F_LOAD_CLIENT_CERT,
                SDC_ECM_R_LOAD_KEY_FAILED);

        goto out;
    }

    if (pcert) {
        *pcert = sdc_ecm_engine_ctx.key_ctx.cert;

        /* user will free the cert pointer, so forget it to avoid giving the same
         * pointer to several users*/
        sdc_ecm_engine_ctx.key_ctx.cert = NULL;
    }

    if (pkey) {
        *pkey = sdc_ecm_engine_ctx.key_ctx.pkey;

        /* user will free the pkey pointer, so forget it to avoid giving the same
         * pointer to several users*/
        sdc_ecm_engine_ctx.key_ctx.pkey = NULL;
    }

    rc = 1;

out:
    /* return 1 if succeed or 0 if error */
    return rc;
}

static int sdc_ecm_load_cert(ENGINE *e, struct sdc_ecm_engine_cmd_load_cert *p)
{
    sdc_ecm_dbg(__FUNCTION__);

    if (!sdc_ecm_engine_ctx.key_ctx.cert)
        sdc_ecm_engine_ctx.key_ctx.ready = 0;

    /* load certificate - only if it is not yet ready */
    if (!sdc_ecm_engine_ctx.key_ctx.ready) {
        int rc = sdc_ecm_dbus_load_key_pair(e, &sdc_ecm_engine_ctx.dbus_ctx,
                    &sdc_ecm_engine_ctx.key_ctx);
        if (!rc) {
            sdc_ecm_err("Failed load cert:sdc_iface_load_key_pair");
            SDC_ECMerr(SDC_ECM_F_LOAD_CERT,
                    SDC_ECM_R_LOAD_KEY_FAILED);

            return 0;
        }
    }

    p->cert = sdc_ecm_engine_ctx.key_ctx.cert;

    /* user will free the cert pointer, so forget it to avoid giving the same
     * pointer to several users*/
    sdc_ecm_engine_ctx.key_ctx.cert = NULL;

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_set_check_cert_mode(const char *p)
{
    int rc = 1;

    sdc_ecm_dbg(__FUNCTION__);

    if (!p) {
        sdc_ecm_err("Invalid check cert mode: NULL");
        SDC_ECMerr(SDC_ECM_F_SET_CHECK_CERT_MODE, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    sdc_ecm_engine_ctx.dbus_ctx.is_server_cert_check_skippable = false;

    if (!strcmp(p, CHECK_CERT_MODE_SERVER_STRING)) {
        sdc_ecm_engine_ctx.check_cert_mode = CHECK_CERT_MODE_SERVER;
    } else if (!strcmp(p, CHECK_CERT_MODE_CHAIN_STRING)) {
        sdc_ecm_engine_ctx.check_cert_mode = CHECK_CERT_MODE_CHAIN;
    } else if (!strcmp(p, CHECK_CERT_MODE_OFF_STRING)) {
        sdc_ecm_engine_ctx.check_cert_mode = CHECK_CERT_MODE_OFF;
        sdc_ecm_engine_ctx.dbus_ctx.is_server_cert_check_skippable = true;
    } else {
        rc = 0;
        sdc_ecm_err(
            "Invalid check cert mode %s, supported: server|chain", p);
        SDC_ECMerr(SDC_ECM_F_SET_CHECK_CERT_MODE, SDC_ECM_R_INVALID_INPUT);
    }

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int sdc_ecm_ctrl_funcs(ENGINE *e, int cmd, long i,
        void *params, void (*f)(void))
{
    (void)e;
    (void)i;
    (void)f;

    int rc = 0;

    sdc_ecm_dbg("%s - cmd=%d", __FUNCTION__, cmd);

    switch (cmd) {

    case LOAD_CERT_CTRL:
        rc = sdc_ecm_load_cert(e, (struct sdc_ecm_engine_cmd_load_cert *)params);
        break;

    case ECM_CHECK_CERT:
        rc = sdc_ecm_dbus_check_server_cert(&sdc_ecm_engine_ctx.dbus_ctx,
                (X509 *)params);
        break;

    case DBUS_BUS_TYPE:
        rc = sdc_ecm_dbus_set_bus_type(&sdc_ecm_engine_ctx.dbus_ctx,
                (const char *)params);
        break;

    case DBUS_BUS_NAME:
        rc = sdc_ecm_dbus_set_bus_name(&sdc_ecm_engine_ctx.dbus_ctx,
                (const char *)params);
        break;

    case DBUS_INTERFACE_NAME:
        rc = sdc_ecm_dbus_set_iface_name(&sdc_ecm_engine_ctx.dbus_ctx,
                (const char *)params);
        break;

    case ECM_CHECK_CERT_MODE:
        rc = sdc_ecm_set_check_cert_mode((const char *)params);
        break;

    case LOGLEVEL:
        rc = update_loglevel(i);
        break;

    case LOGFILENAME:
        rc = update_logfilename((const char *)params);
        break;

    default:
        sdc_ecm_err("Unsupported command");
        SDC_ECMerr(SDC_ECM_F_CTRL, SDC_ECM_R_NOT_SUPPORTED);
        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int sdc_ecm_pkey_meths(ENGINE *e,
    EVP_PKEY_METHOD **meth, const int **nids, int nid)
{
    (void)e;

    sdc_ecm_dbg(__FUNCTION__);

    if (!meth) {
        /* curl tries to get sdc_ecm_pkey_meths for all evp_pkey keys;
         * forbid this by returning 'nothing is supported' if private key and
         * certificate are not loaded */
        if (!sdc_ecm_engine_ctx.key_ctx.ready)
            return 0;

        /* set supported nids */
        *nids = sdc_ecm_pkey_nids;
        return 1;
    }

    if (nid == sdc_ecm_pkey_nids[0]) {
        *meth = sdc_ecm_pkey_rsa;
    } else {
        sdc_ecm_err("Unsupported PKEY method nid: %d", nid);
        SDC_ECMerr(SDC_ECM_F_PKEY_METHS, SDC_ECM_R_NOT_SUPPORTED);
        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_asn1_meths(ENGINE *e,
    EVP_PKEY_ASN1_METHOD **meth, const int **nids, int nid)
{
    (void)e;

    sdc_ecm_dbg(__FUNCTION__);

    if (!meth) {
        /* set supported nids */
        *nids = sdc_ecm_pkey_asn1_nids;
        return 1;
    }

    if (nid == sdc_ecm_pkey_asn1_nids[0]) {
        *meth = sdc_ecm_pkey_asn1_pub;
    } else {
        sdc_ecm_err("Unsupported PKEY method nid: %d", nid);
        SDC_ECMerr(SDC_ECM_F_PKEY_ASN1_METHS, SDC_ECM_R_NOT_SUPPORTED);
        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}

#ifdef NO_SDC_PREHASHED_SIGN
static int sdc_ecm_digests(ENGINE *e, const EVP_MD **digest,
        const int **nids, int nid)
{
    (void)e;

    sdc_ecm_ext_dbg(__FUNCTION__);

    if (!digest) {
        /* set supported nids */
        *nids = sdc_ecm_md_nids;
        return (sizeof(sdc_ecm_md_nids) / sizeof(sdc_ecm_md_nids[0]) - 1);
    }

    switch (nid) {
    case NID_sha1:
        *digest = &sdc_ecm_SHA1_md;
        break;

    case NID_sha256:
        *digest = &sdc_ecm_SHA256_md;
        break;

    case NID_sha512:
        *digest = &sdc_ecm_SHA512_md;
        break;

    default:
        sdc_ecm_err("Unsupported MD method nid: %d", nid);
        SDC_ECMerr(SDC_ECM_F_DIGEST, SDC_ECM_R_NOT_SUPPORTED);
        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}
#endif /* NO_SDC_PREHASHED_SIGN */

static int sdc_ecm_rsa_pub_enc(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding)
{
    (void)flen;
    (void)from;
    (void)to;
    (void)rsa;
    (void)padding;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_rsa_pub_enc");
    SDC_ECMerr(SDC_ECM_F_RSA_PUB_ENC, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_rsa_pub_dec(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding)
{
    (void)flen;
    (void)from;
    (void)to;
    (void)rsa;
    (void)padding;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_rsa_pub_dec");
    SDC_ECMerr(SDC_ECM_F_RSA_PUB_DEC, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_rsa_priv_enc(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding)
{
    (void)flen;
    (void)from;
    (void)to;
    (void)rsa;
    (void)padding;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_rsa_priv_enc");
    SDC_ECMerr(SDC_ECM_F_RSA_PRIV_ENC, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_rsa_priv_dec(int flen, const unsigned char *from,
    unsigned char *to, RSA *rsa, int padding)
{
    (void)flen;
    (void)from;
    (void)to;
    (void)rsa;
    (void)padding;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_rsa_pub_dec");
    SDC_ECMerr(SDC_ECM_F_RSA_PRIV_DEC, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_rsa_sign(int type,
    const unsigned char *m, unsigned int m_length,
    unsigned char *sigret, unsigned int *siglen, const RSA *rsa)
{
    int rc;
    size_t siglen_s = *siglen;
    struct sdc_ecm_dbus_ctx *rsa_dbus_ctx =
            RSA_get_ex_data(rsa, sdc_ecm_rsa_dbus_ctx_ind);

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_inf("Signing with RSA type=%d", type);

    if (!rsa_dbus_ctx->is_server_cert_check_skippable &&
        !rsa_dbus_ctx->is_server_cert_checked) {
        sdc_ecm_err("Forbidden to sign before ECM server certificate check");
        SDC_ECMerr(SDC_ECM_F_RSA_SIGN,
                SDC_ECM_R_FORBIDDEN_SIGN_BEFORE_SERVER_CERT_CHECK);

        return 0;
    }

    rc = sdc_ecm_dbus_sign(rsa_dbus_ctx, type, m, m_length,
            sigret, &siglen_s);
    if (!rc) {
        sdc_ecm_err("Failed sdc_ecm_rsa_sign");
        SDC_ECMerr(SDC_ECM_F_RSA_SIGN, SDC_ECM_R_GATEWAY_SIGN_FAILED);
    }

    *siglen = (unsigned int)siglen_s;

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int sdc_ecm_rsa_verify(int type,
    const unsigned char *m, unsigned int m_length,
    const unsigned char *sigbuf, unsigned int siglen, const RSA *rsa)
{
    (void)type;
    (void)m;
    (void)m_length;
    (void)sigbuf;
    (void)siglen;
    (void)rsa;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_rsa_verify");
    SDC_ECMerr(SDC_ECM_F_RSA_VERIFY, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_rsa_init(RSA *rsa)
{
    sdc_ecm_dbg(__FUNCTION__);

    int rc = RSA_set_ex_data(rsa, sdc_ecm_rsa_dbus_ctx_ind,
                             &sdc_ecm_engine_ctx.dbus_ctx);

    if (!rc) {
        sdc_ecm_err("Failed sdc_ecm_rsa_init: RSA_set_ex_data");
        SDC_ECMerr(SDC_ECM_F_RSA_INIT, SDC_ECM_R_INIT_FAILED);
    }

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int sdc_ecm_rsa_finish(RSA *rsa)
{
    (void)rsa;

    sdc_ecm_dbg(__FUNCTION__);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_init(EVP_PKEY_CTX *ctx)
{
    struct sdc_ecm_pkey_ctx_t *pkey_ctx;

    sdc_ecm_dbg(__FUNCTION__);

    if (!ctx) {
        sdc_ecm_err("Invalid input: sdc_ecm_pkey_init");
        SDC_ECMerr(SDC_ECM_F_PKEY_INIT, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    pkey_ctx = malloc(sizeof(struct sdc_ecm_pkey_ctx_t));
    if (!pkey_ctx) {
        sdc_ecm_err("No mem: sdc_ecm_pkey_init");
        SDC_ECMerr(SDC_ECM_F_PKEY_INIT, SDC_ECM_R_NO_MEM);
        return 0;
    }

    /* init ctx */
    pkey_ctx->md_type = NID_sha256;
    pkey_ctx->dbus_ctx = &sdc_ecm_engine_ctx.dbus_ctx;

    EVP_PKEY_CTX_set_data(ctx, pkey_ctx);

    /* return 1 if succeed or 0 if error */
    /* PRQA: Lint Message 429: pkey_ctx is not freed, and that's correct */
    return 1; /*lint !e429 */
}

static int sdc_ecm_pkey_copy(EVP_PKEY_CTX *dst, EVP_PKEY_CTX *src)
{
    struct sdc_ecm_pkey_ctx_t *pkey_ctx_src;
    struct sdc_ecm_pkey_ctx_t *pkey_ctx_dst;

    sdc_ecm_dbg(__FUNCTION__);

    if (!dst || !src) {
        sdc_ecm_err("Invalid input: sdc_ecm_pkey_copy");
        SDC_ECMerr(SDC_ECM_F_PKEY_COPY, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    /* copy ctx */
    pkey_ctx_src = EVP_PKEY_CTX_get_data(src);
    pkey_ctx_dst = malloc(sizeof(struct sdc_ecm_pkey_ctx_t));
    if (!pkey_ctx_dst) {
        sdc_ecm_err("No mem: sdc_ecm_pkey_copy");
        SDC_ECMerr(SDC_ECM_F_PKEY_COPY, SDC_ECM_R_NO_MEM);
        return 0;
    }

    pkey_ctx_dst->md_type = pkey_ctx_src->md_type;
    pkey_ctx_dst->dbus_ctx = pkey_ctx_src->dbus_ctx;

    EVP_PKEY_CTX_set_data(dst, pkey_ctx_dst);

    /* return 1 if succeed or 0 if error */
    /* PRQA: Lint Message 429: pkey_ctx_dst is not freed, and that's correct */
    return 1; /*lint !e429 */
}

static void sdc_ecm_pkey_cleanup(EVP_PKEY_CTX *ctx)
{
    struct sdc_ecm_pkey_ctx_t *pkey_ctx;

    sdc_ecm_dbg(__FUNCTION__);

    if (!ctx)
        return;

    pkey_ctx = EVP_PKEY_CTX_get_data(ctx);

    free(pkey_ctx);

    sdc_ecm_dbus_cleanup_key_pair(&sdc_ecm_engine_ctx.key_ctx);
}

static int sdc_ecm_pkey_sign_init(EVP_PKEY_CTX *ctx)
{
    (void)ctx;

    sdc_ecm_dbg(__FUNCTION__);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_ctrl(EVP_PKEY_CTX *ctx, int type, int p1, void *p2)
{
    (void)p1;

    struct sdc_ecm_pkey_ctx_t *pkey_ctx;

    sdc_ecm_dbg(__FUNCTION__);

    if (!ctx) {
        sdc_ecm_err("Invalid ctrl inpuit: ctx");
        SDC_ECMerr(SDC_ECM_F_PKEY_CTRL, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    pkey_ctx = EVP_PKEY_CTX_get_data(ctx);
    if (!pkey_ctx) {
        sdc_ecm_err("Invalid ctrl inpuit: pkey_ctx");
        SDC_ECMerr(SDC_ECM_F_PKEY_CTRL, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    switch (type) {

    case EVP_PKEY_CTRL_MD:
        pkey_ctx->md_type = EVP_MD_type((const EVP_MD *)p2);

        sdc_ecm_inf("Set PKey MD type=%d", pkey_ctx->md_type);
        break;

    default:
        sdc_ecm_err("Unsupported PKey ctrl: %d", type);
        SDC_ECMerr(SDC_ECM_F_PKEY_CTRL, SDC_ECM_R_NOT_SUPPORTED);
        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_ctrl_str(EVP_PKEY_CTX *ctx, const char *type,
    const char *value)
{
    sdc_ecm_dbg(__FUNCTION__);

    if (!ctx || !type || !value) {
        sdc_ecm_err("Invalid ctl_str inpuit: ctx");
        SDC_ECMerr(SDC_ECM_F_PKEY_CTRL_STR, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    if (!strcmp(type, "digest")) {
        const EVP_MD *md = EVP_get_digestbyname(value);

        return sdc_ecm_pkey_ctrl(ctx, EVP_PKEY_CTRL_MD, 0, (EVP_MD *)md);
    } else {
        sdc_ecm_err("Unsupported PKey ctrl str: %s", type);
        SDC_ECMerr(SDC_ECM_F_PKEY_CTRL_STR, SDC_ECM_R_NOT_SUPPORTED);
    }

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_pkey_sign(EVP_PKEY_CTX *ctx,
    unsigned char *sig, size_t *siglen,
    const unsigned char *tbs, size_t tbslen)
{
    int rc;
    EVP_PKEY *k;
    struct sdc_ecm_pkey_ctx_t *pkey_ctx;

    sdc_ecm_dbg(__FUNCTION__);

    if (!ctx) {
        sdc_ecm_err("Invalid input: sdc_ecm_pkey_sign");
        SDC_ECMerr(SDC_ECM_F_PKEY_SIGN, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    pkey_ctx = EVP_PKEY_CTX_get_data(ctx);
    if (!pkey_ctx) {
        sdc_ecm_err("Invalid sign input: pkey_ctx");
        SDC_ECMerr(SDC_ECM_F_PKEY_SIGN, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    k = EVP_PKEY_CTX_get0_pkey(ctx);
    if (!k) {
        sdc_ecm_err("Failed EVP_PKEY_CTX_get0_pkey");
        SDC_ECMerr(SDC_ECM_F_PKEY_SIGN, SDC_ECM_R_INVALID_INPUT);

        return 0;
    }

    if (k->type != EVP_PKEY_RSA) {
        sdc_ecm_err("Invalid key type %d\n", k->type);
        SDC_ECMerr(SDC_ECM_F_PKEY_SIGN, SDC_ECM_R_INVALID_KEY_TYPE);

        return 0;
    }

    sdc_ecm_inf("Signing with PKEY RSA type=%d", pkey_ctx->md_type);

    if (!pkey_ctx->dbus_ctx->is_server_cert_check_skippable &&
        !pkey_ctx->dbus_ctx->is_server_cert_checked) {
        sdc_ecm_err("Forbidden to sign before ECM server certificate check");
        SDC_ECMerr(SDC_ECM_F_PKEY_SIGN,
                SDC_ECM_R_FORBIDDEN_SIGN_BEFORE_SERVER_CERT_CHECK);

        return 0;
    }

    rc = sdc_ecm_dbus_sign(pkey_ctx->dbus_ctx, pkey_ctx->md_type,
                        tbs, tbslen, sig, siglen);
    if (!rc) {
        sdc_ecm_err("Failed sdc_ecm_rsa_sign");
        SDC_ECMerr(SDC_ECM_F_PKEY_SIGN, SDC_ECM_R_GATEWAY_SIGN_FAILED);

        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_verify_init(EVP_PKEY_CTX *ctx)
{
    (void)ctx;

    sdc_ecm_dbg(__FUNCTION__);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_verify(EVP_PKEY_CTX *ctx,
    const unsigned char *sig, size_t siglen,
    const unsigned char *tbs, size_t tbslen)
{
    (void)ctx;
    (void)sig;
    (void)siglen;
    (void)tbs;
    (void)tbslen;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_pkey_verify");
    SDC_ECMerr(SDC_ECM_F_PKEY_VERIFY, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_pkey_encrypt_init(EVP_PKEY_CTX *ctx)
{
    (void)ctx;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_pkey_encrypt_init");
    SDC_ECMerr(SDC_ECM_F_PKEY_ENCRYPT_INIT, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_pkey_encrypt(EVP_PKEY_CTX *ctx,
    unsigned char *out, size_t *outlen,
    const unsigned char *in, size_t inlen)
{
    (void)ctx;
    (void)out;
    (void)outlen;
    (void)in;
    (void)inlen;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_pkey_encrypt");
    SDC_ECMerr(SDC_ECM_F_PKEY_ENCRYPT, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_pkey_decrypt_init(EVP_PKEY_CTX *ctx)
{
    (void)ctx;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_pkey_decrypt_init");
    SDC_ECMerr(SDC_ECM_F_PKEY_DECRYPT_INIT, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_pkey_decrypt(EVP_PKEY_CTX *ctx,
    unsigned char *out, size_t *outlen,
    const unsigned char *in, size_t inlen)
{
    (void)ctx;
    (void)out;
    (void)outlen;
    (void)in;
    (void)inlen;

    sdc_ecm_dbg(__FUNCTION__);

    sdc_ecm_err("Not supported: sdc_ecm_pkey_decrypt");
    SDC_ECMerr(SDC_ECM_F_PKEY_DECRYPT, SDC_ECM_R_NOT_SUPPORTED);

    /* return 1 if succeed or 0 if error */
    return 0;
}

static int sdc_ecm_pkey_asn1_pub_cmp(const EVP_PKEY *a, const EVP_PKEY *b)
{
    RSA *rsa_a;
    RSA *rsa_b;
    struct sdc_ecm_dbus_ctx *dbus_ctx_a;
    struct sdc_ecm_dbus_ctx *dbus_ctx_b;

    sdc_ecm_dbg(__FUNCTION__);

    if (!a || !b) {
        sdc_ecm_err("Invalid input: sdc_ecm_pkey_asn1_pub_cmp");
        SDC_ECMerr(SDC_ECM_F_PKEY_ASN1_PUB_CMP, SDC_ECM_R_INVALID_INPUT);

        return 0;
    }

    if (a->type != EVP_PKEY_RSA || b->type != EVP_PKEY_RSA) {
        sdc_ecm_err("Invalid key type: a=%d, b=%d", a->type, b->type);
        SDC_ECMerr(SDC_ECM_F_PKEY_ASN1_PUB_CMP, SDC_ECM_R_INVALID_KEY_TYPE);

        return 0;
    }

    rsa_a = a->pkey.rsa;
    rsa_b = b->pkey.rsa;
    if (!rsa_a || !rsa_b) {
        sdc_ecm_err("NULL key RSA");
        SDC_ECMerr(SDC_ECM_F_PKEY_ASN1_PUB_CMP, SDC_ECM_R_INVALID_KEY_TYPE);

        return 0;
    }

    dbus_ctx_a = RSA_get_ex_data(rsa_a, sdc_ecm_rsa_dbus_ctx_ind);
    dbus_ctx_b = RSA_get_ex_data(rsa_b, sdc_ecm_rsa_dbus_ctx_ind);

    if (!dbus_ctx_a || dbus_ctx_a != dbus_ctx_b) {
        sdc_ecm_err("PKey ctx not equal: sdc_ecm_pkey_asn1_pub_cmp");
        SDC_ECMerr(SDC_ECM_F_PKEY_ASN1_PUB_CMP, SDC_ECM_R_INVALID_KEY_ID);

        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_asn1_pub_dec(EVP_PKEY *pk, X509_PUBKEY *pub)
{
    sdc_ecm_dbg(__FUNCTION__);

    if (!pub || !pk)
        return 0;

    /* don't actually decode;
     * this function exists to create certificate-key pair  */

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_pkey_asn1_pub_enc(X509_PUBKEY *pub, const EVP_PKEY *pk)
{
    sdc_ecm_dbg(__FUNCTION__);

    if (!pub || !pk)
        return 0;

    /* don't actually encode;
     * this function exists to create certificate-key pair  */

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_init(ENGINE *e)
{
    (void)e;

    sdc_ecm_inf(__FUNCTION__);

    sdc_ecm_dbg("sdc_ecm_init started");

    sdc_ecm_rsa_dbus_ctx_ind = -1;
    sdc_ecm_engine_ctx.check_cert_mode = CHECK_CERT_MODE_SERVER;
    memset(&sdc_ecm_engine_ctx.dbus_ctx, 0, sizeof(struct sdc_ecm_dbus_ctx));

    ERR_load_SDC_ECM_strings();
    ERR_clear_error();

    if (!sdc_ecm_pkey_meth_init()) {
        sdc_ecm_err("Failed sdc_ecm_pkey_meth_init");
        SDC_ECMerr(SDC_ECM_F_INIT, SDC_ECM_R_INIT_FAILED);
        return 0;
    }

    if (!sdc_ecm_pkey_asn1_meth_init()) {
        sdc_ecm_err("Failed sdc_ecm_pkey_asn1_meth_init");
        SDC_ECMerr(SDC_ECM_F_INIT, SDC_ECM_R_INIT_FAILED);
        sdc_ecm_pkey_meth_destroy();
        return 0;
    }

    if (!sdc_ecm_dbus_init(&sdc_ecm_engine_ctx.dbus_ctx,
            &sdc_ecm_engine_ctx.key_ctx)) {
        sdc_ecm_err("Failed sdc_ecm_dbus_init");
        SDC_ECMerr(SDC_ECM_F_INIT, SDC_ECM_R_INIT_FAILED);
        return 0;
    }

    sdc_ecm_rsa_dbus_ctx_ind = RSA_get_ex_new_index(0, "sdc-ecm DBus ctx",
                                                  NULL, NULL, NULL);
    if (sdc_ecm_rsa_dbus_ctx_ind < 0) {
        sdc_ecm_err("Failed RSA_get_ex_new_index");
        SDC_ECMerr(SDC_ECM_F_INIT, SDC_ECM_R_INIT_FAILED);
        return 0;
    }

    sdc_ecm_dbg("sdc_ecm_init succeed");

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_destroy(ENGINE *e)
{
    (void)e;

    sdc_ecm_inf(__FUNCTION__);

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_finish(ENGINE *e)
{
    (void)e;

    sdc_ecm_inf(__FUNCTION__);

    sdc_ecm_dbg("sdc_ecm_finish started");

    sdc_ecm_dbus_finish(&sdc_ecm_engine_ctx.dbus_ctx,
            &sdc_ecm_engine_ctx.key_ctx);

    sdc_ecm_pkey_meth_destroy();

    sdc_ecm_pkey_asn1_meth_destroy();

    ERR_unload_SDC_ECM_strings();

    sdc_ecm_dbg("sdc_ecm_finish succeed");

    /* return 1 if succeed or 0 if error */
    return 1;
}

static int sdc_ecm_bind(ENGINE *e, const char *id)
{
    update_loglevel(DEF_LOGLEVEL);
    update_logfilename(DEF_LOGFILENAME);

    sdc_ecm_inf(__FUNCTION__);

    if (strcmp(id, sdc_ecm_id)) {
        sdc_ecm_err("Invalid engine ID");
        SDC_ECMerr(SDC_ECM_F_BIND, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    if (!e) {
        sdc_ecm_err("Invalid input parameter: engine");
        SDC_ECMerr(SDC_ECM_F_BIND, SDC_ECM_R_INVALID_INPUT_ENGINE);
        return 0;
    }

    if (!ENGINE_set_id(e, sdc_ecm_id) ||
        !ENGINE_set_name(e, sdc_ecm_name) ||
        !ENGINE_set_init_function(e, sdc_ecm_init) ||
        !ENGINE_set_destroy_function(e, sdc_ecm_destroy) ||
        !ENGINE_set_finish_function(e, sdc_ecm_finish) ||
        !ENGINE_set_RSA(e, &sdc_ecm_rsa) ||
        !ENGINE_set_pkey_meths(e, sdc_ecm_pkey_meths) ||
        !ENGINE_set_pkey_asn1_meths(e, sdc_ecm_pkey_asn1_meths) ||
        !ENGINE_set_flags(e, ENGINE_FLAGS_NO_REGISTER_ALL) ||
#ifdef NO_SDC_PREHASHED_SIGN
        !ENGINE_set_digests(e, sdc_ecm_digests) ||
        !ENGINE_set_default_digests(e) ||
#endif /* NO_SDC_PREHASHED_SIGN */
        !ENGINE_set_load_privkey_function(e, sdc_ecm_load_privkey) ||
        !ENGINE_set_load_pubkey_function(e, sdc_ecm_load_pubkey) ||
        !ENGINE_set_load_ssl_client_cert_function(e,
            sdc_ecm_load_client_cert) ||
        !ENGINE_set_ctrl_function(e, sdc_ecm_ctrl_funcs) ||
        !ENGINE_set_cmd_defns(e, sdc_ecm_cmd_defns)) {
        sdc_ecm_err("Engine binding failed");
        SDC_ECMerr(SDC_ECM_F_BIND, SDC_ECM_R_INIT_FAILED);
        return 0;
    }

    /* return 1 if succeed or 0 if error */
    return 1;
}

IMPLEMENT_DYNAMIC_BIND_FN(sdc_ecm_bind)
IMPLEMENT_DYNAMIC_CHECK_FN()
